/*
 * Copyright 2018-2019, 2025 NXP
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include <xtensa/config/core.h>
#include <xtensa/xos.h>
#include <stdio.h>

#include "fsl_device_registers.h"
#include "fsl_debug_console.h"
#include "pin_mux.h"
#include "board.h"
#include "shared_memory_def.h"
#include "fsl_inputmux.h"

#include "rpmsg_lite.h"
#include "rpmsg_queue.h"
#include "rpmsg_ns.h"

/*******************************************************************************
 * Definitions
 ******************************************************************************/

#define HIFI4_NS_ANNOUNCE_STRING "Compute - HiFi4"

#define DSP_THREAD_PRIORITY (1)
#define DSP_THREAD_STACK_SIZE (1024)
static uint8_t dsp_thread_stack[DSP_THREAD_STACK_SIZE];
/*******************************************************************************
 * Prototypes
 ******************************************************************************/
int DSP_Main(void *arg, int wake_value);

static void XOS_Init(void)
{
    xos_set_clock_freq(XOS_CLOCK_FREQ);
    xos_start_system_timer(-1, 0);
}

#define AT_DSPCORE_SECTION_INIT(var) __attribute__((section("dsp_core.data"))) var
#define AT_DSPCORE_SECTION(var) __attribute__((section("dsp_core.bss"))) var

/*******************************************************************************
 * Code
 ******************************************************************************/
/*!
 * @brief Main function
 */
int main(void)
{
    XosThread thread_main;

    AT_DSPCORE_SECTION_INIT(volatile static int dsp_core_data = 1);
    AT_DSPCORE_SECTION(volatile static int dsp_core_bss);

    CLOCK_EnableClock(kCLOCK_InputMux);
    INPUTMUX_Init(INPUTMUX0);
    INPUTMUX_AttachSignal(INPUTMUX0, 1U, kINPUTMUX_Mu4BToDspInterrupt); /* MU4B for HiFi4 DSP */
    INPUTMUX_Deinit(INPUTMUX0);

    XOS_Init();
    /* Init board hardware. */
    BOARD_InitBootPins();
    DbgConsole_Init_Alt(BOARD_DEBUG_UART_INSTANCE, kSerialPort_Uart);

    LED_GREEN_INIT(LOGIC_LED_OFF);
    PRINTF("[HiFi4] Running\r\n");
    dsp_core_bss = dsp_core_data;

    xos_thread_create(&thread_main, NULL, DSP_Main, NULL, "DSP Main", dsp_thread_stack, DSP_THREAD_STACK_SIZE,
                      DSP_THREAD_PRIORITY, 0, 0);

    /* Start XOS scheduler - does not return */
    xos_start(0);
}

static void app_nameservice_isr_cb(uint32_t new_ept, const char *new_ept_name, uint32_t flags, void *user_data)
{
    uint32_t *data = (uint32_t *)user_data;
    if(data) {
        *data = new_ept;
    }
}

int DSP_Main(void *arg, int wake_value)
{
    struct rpmsg_lite_instance * p_hifi4_cpu0_rpmsg;
    struct rpmsg_lite_endpoint *p_hifi4_cpu0_ep = NULL;
    rpmsg_queue_handle hifi4_cpu0_queue;
    rpmsg_ns_handle cpu0_ns_handle;
	cpu0_hifi4_parcel_t cpu0_hifi4_parcel;
	hifi4_cpu0_parcel_t hifi4_cpu0_parcel;
	uint32_t response_counter = 0;
	int32_t status;

    PRINTF("[HiFi4] - DSP Main\r\n");

    p_hifi4_cpu0_rpmsg = rpmsg_lite_remote_init((void *)CPU0_HIFI4_RPMSG_LITE_BASE,
                                                        RL_PLATFORM_IMXRT700_M33_0_HIFI4_LINK_ID, RL_NO_FLAGS);

    hifi4_cpu0_queue = rpmsg_queue_create(p_hifi4_cpu0_rpmsg);
    p_hifi4_cpu0_ep = rpmsg_lite_create_ept(p_hifi4_cpu0_rpmsg, HIFI4_EPT_ADDR, rpmsg_queue_rx_cb, hifi4_cpu0_queue);
    cpu0_ns_handle = rpmsg_ns_bind(p_hifi4_cpu0_rpmsg, app_nameservice_isr_cb, ((void *)0));
    (void)cpu0_ns_handle;
    rpmsg_lite_wait_for_link_up(p_hifi4_cpu0_rpmsg, RL_BLOCK);
    PRINTF("[HiFi4] Link up!\r\n");

    xos_thread_sleep_msec(100);
	rpmsg_ns_announce(p_hifi4_cpu0_rpmsg, p_hifi4_cpu0_ep, HIFI4_NS_ANNOUNCE_STRING, (uint32_t)RL_NS_CREATE);
    xos_thread_sleep_msec(10);
	PRINTF("[HiFi4] Name service announce sent.\r\n");

    while (1)
    {
        status = rpmsg_queue_recv(p_hifi4_cpu0_rpmsg, hifi4_cpu0_queue, NULL, (char *)&cpu0_hifi4_parcel, 
                                  sizeof(cpu0_hifi4_parcel), NULL, RL_BLOCK);
     	if (status != RL_SUCCESS)
     	{
     		PRINTF("Failed to get item from RPMsg queue.\r\n");
     	} else {
            PRINTF("[HiFi4] RECV %s\r\n", (char*)cpu0_hifi4_parcel.data);
        }

        if(response_counter % 3 == 1)
        {
            LED_GREEN_ON();
        }
        else
        {
            LED_GREEN_OFF();
        }
  
        snprintf((char *)hifi4_cpu0_parcel.data, 16, "H4->C0:%d", response_counter++);
        
        rpmsg_lite_send(p_hifi4_cpu0_rpmsg, p_hifi4_cpu0_ep, CPU0_EPT_ADDR,
			            (char *)hifi4_cpu0_parcel.data, sizeof(hifi4_cpu0_parcel), RL_BLOCK);
    }
}
